package edu.uky.ai.util;

import java.util.ArrayList;

/**
 * A more convenient representation of command line argument based on keys and
 * values. A key is an argument that starts with a dash "-" character. Every
 * other argument is considered a value.
 * 
 * @author Stephen G. Ware
 */
public class Arguments {

	/**
	 * Arguments, divided into groups. Each group (except the first) begins
	 * with a key and contains all the values that follow that key. The first
	 * group is all values before the first key.
	 */
	public final String[][] groups;
	
	/**
	 * Constructs a new group of argument from an array of strings.
	 * 
	 * @param args the command line arguments
	 */
	public Arguments(String[] args) {
		ArrayList<String[]> groups = new ArrayList<>();
		ArrayList<String> group = new ArrayList<>();
		for(int i=0; i<args.length; i++) {
			if(args[i].startsWith("-")) {
				if(group.size() != 0)
					groups.add(group.toArray(new String[group.size()]));
				group.clear();
			}
			group.add(args[i]);
		}
		if(group.size() != 0)
			groups.add(group.toArray(new String[group.size()]));
		this.groups = groups.toArray(new String[groups.size()][]);
	}
	
	/**
	 * Checks whether a given key is present in the arguments.
	 * 
	 * @param key the key to look for, including the initial dash
	 * @return true if the key is present, false otherwise
	 */
	public boolean containsKey(String key) {
		for(String[] group : groups)
			if(group[0].equals(key))
				return true;
		return false;
	}
	
	/**
	 * Returns the value immediately after a given key.
	 * 
	 * @param key the key to look for, including the initial dash 
	 * @return the value following the key
	 * @throws IllegalArgumentException if the key is not present, if there is
	 * no value after the key, or if there is more than one value after the key
	 */
	public String getValue(String key) {
		for(String[] group : groups) {
			if(group[0].equals(key)) {
				if(group.length != 2)
					throw new IllegalArgumentException("Expected exactly one value after the key \"" + key + "\".");
				else
					return group[1];
			}
		}
		throw new IllegalArgumentException("No value provided for the key \"" + key + "\".");
	}
	
	/**
	 * Returns the values (0, 1, or many) immediately after a given key.
	 * 
	 * @param key the key to look for, including the initial dash 
	 * @return the values following the key
	 * @throws IllegalArgumentException if the key is not present
	 */
	public String[] getValues(String key) {
		for(String[] group : groups) {
			if(group[0].equals(key)) {
				String[] values = new String[group.length - 1];
				System.arraycopy(group, 1, values, 0, values.length);
				return values;
			}
		}
		throw new IllegalArgumentException("No values provided for the key \"" + key + "\".");
	}
}
