package edu.uky.ai.data;

import edu.uky.ai.Settings;
import edu.uky.ai.util.ImmutableArray;

/**
 * An individual observation in a {@link LabeledDataSet}, with a value for each
 * {@link Feature} and for the data set's class label.
 * 
 * @author Stephen G. Ware
 */
public class LabeledDataPoint extends DataPoint {

	/** Serial verison UID */
	private static final long serialVersionUID = Settings.VERSION_UID;
	
	/** The labeled data set object this data point belongs to */
	public final LabeledDataSet set;
	
	/** The value of the data set's class label for this point */
	public final Value label;
	
	/**
	 * Constructs a new labeled data point.
	 * 
	 * @param set the labeled data set to which this point belongs
	 * @param values the values for each feature
	 * @param label the value of the data set's class label for this point
	 */
	LabeledDataPoint(LabeledDataSet set, ImmutableArray<Value> values, Value label) {
		super(set, values);
		this.set = set;
		this.label = label;
	}
	
	@Override
	public String toString() {
		return super.toString() + " = " + label;
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public <V extends Value> V get(Feature<V> feature) {
		if(feature.equals(set.labels))
			return (V) label;
		else
			return super.get(feature);
	}
	
	/**
	 * Removes the class label from this data point, returning an unlabeled
	 * {@link DataPoint}.
	 * 
	 * @return the unlabeled data point
	 */
	public DataPoint removeLabel() {
		return new DataPoint(set, values);
	}
	
	@Override
	LabeledDataPoint associate(DataSet set) {
		return new LabeledDataPoint((LabeledDataSet) set, values, label);
	}
	
	@Override
	LabeledDataPoint remove(Feature<?> feature) {
		return new LabeledDataPoint(set, super.remove(feature).values, label);
	}
}