package edu.uky.ai.data;

import java.io.Serializable;
import java.util.ArrayList;

import edu.uky.ai.Settings;
import edu.uky.ai.util.ImmutableArray;

/**
 * An individual observation in a {@link DataSet}, with a value for each
 * {@link Feature}.
 * 
 * @author Stephen G. Ware
 */
public class DataPoint implements Serializable {

	/** Serial verison UID */
	private static final long serialVersionUID = Settings.VERSION_UID;
	
	/** The data set object this data point belongs to */
	public final DataSet set;
	
	/** The list of values, one for each {@link Feature} in the point's data set */
	public final ImmutableArray<Value> values;
	
	/**
	 * Constructs a new data point.
	 * 
	 * @param set the data set to which this point belongs
	 * @param values the values for each feature
	 */
	DataPoint(DataSet set, ImmutableArray<Value> values) {
		this.set = set;
		this.values = values;
	}
	
	@Override
	public String toString() {
		return values.toString();
	}
	
	/**
	 * Returns this data point's value for the given feature.
	 * 
	 * @param <V> the value type of the feature
	 * @param feature the feature
	 * @return the value for that feature
	 */
	@SuppressWarnings("unchecked")
	public <V extends Value> V get(Feature<V> feature) {
		int index = set.features.indexOf(feature);
		if(index == -1)
			throw new IllegalArgumentException("The feature \"" + feature.name + "\" does not appear in this data set.");
		return (V) values.get(index);
	}
	
	/**
	 * Creates a new data point object associated with the given data set.
	 * 
	 * @param set the data set which which to associate
	 * @return the new data point
	 */
	DataPoint associate(DataSet set) {
		return new DataPoint(set, values);
	}
	
	/**
	 * Removes a value corresponding to a given feature.
	 * 
	 * @param feature the feature
	 * @return the data point with value removed
	 */
	DataPoint remove(Feature<?> feature) {
		ArrayList<Value> values = new ArrayList<>();
		for(Feature<?> f : set.features)
			if(!f.equals(feature))
				values.add(get(f));
		return new DataPoint(set, new ImmutableArray<>(values.toArray(new Value[values.size()])));
	}
}