package edu.uky.ai.planning.ps;

import edu.uky.ai.logic.Bindings;
import edu.uky.ai.logic.ListBindings;
import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.Substitution;
import edu.uky.ai.planning.Problem;
import edu.uky.ai.util.DirectedAcyclicGraph;
import edu.uky.ai.util.DirectedEdge;
import edu.uky.ai.util.ImmutableList;

/**
 * This class represents an specific partial plan in a plan-space search.
 * 
 * @author Stephen G. Ware
 */
public class PlanSpaceNode implements Partial {

	/** This node's parent (the node of which this node is a refinement) */
	public final PlanSpaceNode parent;
	
	/** The steps in this plan */
	public final ImmutableList<PartialStep> steps;
	
	/** The binding constraints specifying which variables map to which values */
	public final Bindings bindings;
	
	/** The constraints on how steps are ordered */
	public final DirectedAcyclicGraph<PartialStep> orderings;
	
	/** The causal links describing how step preconditions are satisfied */
	public final ImmutableList<CausalLink> causalLinks;
	
	/** A list of flaws (if any) that prevent this plan from being a solution */
	public final ImmutableList<Flaw> flaws;
	
	/** Records whether or not this step has been visited during the search */
	private boolean visited = false;
	
	/**
	 * Constructs a root node for a given planning problem.  This constructor
	 * should only be used by {@link PlanSpaceRoot}.
	 * 
	 * @param problem the problem to solve
	 */
	PlanSpaceNode(Problem problem) {
		parent = null;
		PartialStep start = new PartialStep(Proposition.TRUE, problem.initial.toProposition());
		PartialStep end = new PartialStep(problem.goal, Proposition.TRUE);
		this.steps = new ImmutableList<PartialStep>().add(start).add(end);
		this.bindings = ListBindings.EMPTY;
		this.orderings = new DirectedAcyclicGraph<PartialStep>().add(start, end);
		this.causalLinks = new ImmutableList<>();
		ImmutableList<Flaw> flaws = new ImmutableList<>();
		for(Literal goal : end.preconditions)
			flaws = flaws.add(new OpenPreconditionFlaw(end, goal));
		this.flaws = flaws;
	}
	
	/**
	 * Constructs a new plan space node with the given steps, bindings,
	 * orderings, causal links, and flaws.
	 * 
	 * @param parent this plan's parent node
	 * @param steps the steps in this plan
	 * @param bindings the binding constraints
	 * @param orderings the orderings between steps
	 * @param causalLinks the causal links
	 * @param flaws a list of flaws
	 */
	private PlanSpaceNode(PlanSpaceNode parent, ImmutableList<PartialStep> steps, Bindings bindings, DirectedAcyclicGraph<PartialStep> orderings, ImmutableList<CausalLink> causalLinks, ImmutableList<Flaw> flaws) {
		this.parent = parent;
		this.steps = steps;
		this.bindings = bindings;
		this.orderings = orderings;
		this.causalLinks = causalLinks;
		this.flaws = flaws;
		PlanSpaceRoot root = getRoot();
		if(!parent.visited) {
			root.budget.incrementOperations();
			root.budget.checkTime();
			parent.visited = true;
			root.visited++;
		}
		root.generated++;
	}
	
	@Override
	public String toString() {
		return toString(bindings);
	}
	
	@Override
	public String toString(Substitution substitution) {
		String str = "===== PLAN SPACE NODE =====";
		str += "\n       Steps: ";
		boolean first = true;
		for(PartialStep step : orderings) {
			if(first)
				first = false;
			else
				str += "\n              ";
			str += step.toString(substitution);
		}
		str += "\n    Bindings: " + bindings.toString().replace("; ", "\n              ");
		str += "\n   Orderings: ";
		first = true;
		for(DirectedEdge<PartialStep> edge : orderings.edges()) {
			if(first)
				first = false;
			else
				str += "\n              ";
			str += edge.tail.toString(substitution) + " < " + edge.head.toString(substitution);
		}
		str += "\nCausal Links: ";
		first = true;
		for(CausalLink link : causalLinks) {
			if(first)
				first = false;
			else
				str += "\n              ";
			str += link.toString(substitution);
		}
		str += "\n       Flaws: ";
		first = true;
		for(Flaw flaw : flaws) {
			if(first)
				first = false;
			else
				str += "\n              ";
			str += flaw.toString(substitution);
		}
		return str;
	}
	
	/**
	 * Returns to {@link PlanSpaceRoot root node} of this search space.
	 * 
	 * @return the root node
	 */
	public PlanSpaceRoot getRoot() {
		PlanSpaceNode current = this;
		while(current.parent != null)
			current = current.parent;
		return (PlanSpaceRoot) current;
	}
	
	/**
	 * Creates a child partial plan node (a node for this this node is the
	 * parent) with the given steps, bindings, orderings, causal links, and
	 * flaws.
	 * 
	 * @param steps the steps in the child plan
	 * @param bindings the bindings in the child plan
	 * @param orderings the orderings between steps in the child plan
	 * @param causalLinks the causal links in the child plan
	 * @param flaws the flaws in the child plan
	 * @return a child partial plan node
	 */
	public PlanSpaceNode expand(ImmutableList<PartialStep> steps, Bindings bindings, DirectedAcyclicGraph<PartialStep> orderings, ImmutableList<CausalLink> causalLinks, ImmutableList<Flaw> flaws) {
		return new PlanSpaceNode(this, steps, bindings, orderings, causalLinks, flaws);
	}
}
