package edu.uky.ai.planning.pg;

import java.util.ArrayList;
import java.util.Iterator;

import edu.uky.ai.logic.Literal;
import edu.uky.ai.planning.Step;

/**
 * Represents a unique step, or action, in a plan graph.  Step nodes have
 * preconditions ({@link LiteralNode literal nodes} which must be ture before
 * this step can happen) and effects ({@link LiteralNode literal nodes} which
 * become true after this step happens). 
 * 
 * @author Stephen G. Ware
 */
public class StepNode extends Node {

	/** The step represented by this node */
	public final Step step;
	
	/** Whether or not this is a dummy persistence step */
	public final boolean persistence;
	
	/** All literal nodes which must be true before this step can happen */
	protected final ArrayList<LiteralNode> preconditions = new ArrayList<>();
	
	/** All literal nodes which become true after this step happens */
	protected final ArrayList<LiteralNode> effects = new ArrayList<>();
	
	/** Tracks how many of this step's preconditions has been met */
	private int literalCount = 0;
	
	/**
	 * Constructs a new step node for the given literal in the given plan
	 * graph.
	 * 
	 * @param graph the graph in which this node will exist
	 * @param step the step this node represents
	 */
	protected StepNode(PlanGraph graph, Step step) {
		super(graph);
		this.step = step;
		this.persistence = false;
	}
	
	/**
	 * Constructs a dummy persistence step for a given literal (i.e. a step
	 * which has only this literal as a precondition and only this literal as
	 * an effect).
	 * 
	 * @param graph the graph in which this node will exist
	 * @param literal the literal which is this step's only precondition and effect
	 */
	protected StepNode(PlanGraph graph, Literal literal) {
		super(graph);
		this.step = new Step("(persist " + literal + ")", literal, literal);
		this.persistence = true;
	}
	
	@Override
	public int hashCode() {
		return step.hashCode();
	}
	
	@Override
	public String toString() {
		return step.toString();
	}
	
	/**
	 * Notifies this step that one of its preconditions has been met.
	 */
	protected void incrementLiteralCount() {
		markForReset();
		literalCount++;
		if(literalCount == preconditions.size())
			graph.nextSteps.add(this);
	}
	
	@Override
	protected boolean setLevel(int level) {
		if(super.setLevel(level)) {
			for(LiteralNode effect : effects)
				effect.setLevel(level);
			return true;
		}
		else
			return false;
	}
	
	@Override
	protected void reset() {
		super.reset();
		literalCount = 0;
	}
	
	/**
	 * Returns all literals at the previous level of the plan graph which are
	 * preconditions of this step.  Note that when this method is called for
	 * level n, it returns steps which exist at level n - 1.
	 * 
	 * @param level the index of a level in the plan graph at which this node exists
	 * @return all literals which exist at the previous level and which are preconditions of this step
	 * @throws IllegalArgumentException if this step does not exist at the given level
	 */
	public Iterable<LiteralNode> getPreconditions(int level) {
		if(!exists(level))
			throw new IllegalArgumentException(this + " does not exist at level " + level + ".");
		return new Iterable<LiteralNode>() {
			@Override
			public Iterator<LiteralNode> iterator() {
				return new NodeIterator<>(level - 1, preconditions);
			}
		};
	}
	
	/**
	 * Returns all literals at a given level of the plan graph which are
	 * effects of this step.
	 * 
	 * @param level the index of a level in the plan graph
	 * @return all literals which exist at that level and which are effects of this step
	 * @throws IllegalArgumentException if this step does not exist at the given level
	 */
	public Iterable<LiteralNode> getEffects(int level) {
		if(!exists(level))
			throw new IllegalArgumentException(this + " does not exist at level " + level + ".");
		return new Iterable<LiteralNode>() {
			@Override
			public Iterator<LiteralNode> iterator() {
				return new NodeIterator<>(level, effects);
			}
		};
	}
}
