package edu.uky.ai.planning.pg;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.planning.Planner;
import edu.uky.ai.planning.Problem;

/**
 * A planner which searches a {@link PlanGraph} for a subgraph that represents
 * a solution to a planning problem.  All such planners share a two-phase
 * search process: First, the plan graph is extended, then it is searched for
 * a solution.  If no solution is found, the graph is extended farther and
 * searched again, and so on, until a solution is found or the planner runs out
 * of resources.
 * 
 * @author Stephen G. Ware
 */
public abstract class PlanGraphPlanner extends Planner<PlanGraphSearch> {

	/**
	 * Constructs a new plan graph planner with the given name.
	 * 
	 * @param name the name of the planner
	 */
	public PlanGraphPlanner(String name) {
		super(name);
	}
	
	@Override
	protected PlanGraphSearch makeSearch(Problem problem, SearchBudget budget) {
		return new PlanGraphSearch(this, problem, budget);
	}
	
	/**
	 * Creates a {@link SubgraphSearch} for the second phase of the planner's
	 * search process.  By the time this method is called, the plan graph will
	 * already have been extended until all goals are non-mutex.  This method
	 * may be called multiple times per problem, but each time the plan graph
	 * will have been extended farther.
	 * 
	 * @param graph the plan graph to search for a solution
	 * @param budget the search budget, which specifies the maximum number of
	 * nodes that may be visited and maximum amount of time that may be spent
	 * @return the second phase of the search process
	 */
	protected abstract SubgraphSearch makeSearch(PlanGraph graph, SearchBudget budget);
}
