package edu.uky.ai.planning.pg;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.LinkedHashMap;

import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.State;
import edu.uky.ai.planning.Problem;
import edu.uky.ai.planning.Step;
import edu.uky.ai.planning.Utilities;
import edu.uky.ai.planning.ss.StateSpaceProblem;

/**
 * <p> plan graph is a data structure that compresses the state space of a
 * planning problem into something that represents possible future states.
 * It is a directed, leveled graph with two kinds of nodes:</p>
 * <p>A {@link LiteralNode LiteralNode} represents a fact:</p>
 * <ul>
 * <li>A literal node exists at level 0 iff it is true in the initial state.
 * </li>
 * <li>A literal node exists at level n &gt; 0 if it exists at level n.</li>
 * <li>A literal node exists at level n &gt; 0 if there exists a step node a level
 * n - 1 which has that literal as an effect.</li>
 * </ul>
 * <p>A {@link StepNode StepNode} represents a step:</p>
 * <ul>
 * <li>No step nodes exist at level 0.</li>
 * <li>A step node exists at level n &gt; 0 if all its preconditions appear at 
 * level n - 1.</li>
 * </ul>
 * <p>Node that a plan graph must be {@link #initialize(State) initialized}
 * to some state before it can be used.  This method resets the plan graph
 * and determines which literals are true at level 0.</p>
 * <p>In order to save a significant amount of memory, there is only 1 object
 * for each literal node and each step node (rather than 1 per level at which
 * that literal or step appears).  A {@link Node node} is simply marked with
 * the earliest level at which it appears, since it will continue to appear
 * at all future levels.  If a literal or step has not appeared yet, it's
 * level is set to -1 and it is said not to exist (even though it still resides
 * in memory for possible later use).</p>
 * <p>Note that a plan graph can be built with or without persistence steps
 * and with or without mutexes.</p>
 * <p>Note that this implementation only computes static mutexes (ones that
 * must always exist).  It does not compute mytexes for competing needs or
 * inconsistent support.</p>
 * 
 * @author Stephen G. Ware
 */
public class PlanGraph {

	/** The planning problem represented by this plan graph */
	public final StateSpaceProblem problem;
	
	/** The literal nodes for the problem's goals */
	public final Iterable<LiteralNode> goals;
	
	/** Whether or not mutexes should be computed */
	private final boolean mutexes;
	
	/** A mapping of literal objects to their corresponding nodes */
	protected final LinkedHashMap<Literal, LiteralNode> literalMap = new LinkedHashMap<>();
	
	/** A mapping of step objects to their corresponding nodes */
	protected final LinkedHashMap<Step, StepNode> stepMap = new LinkedHashMap<>();

	/** A list of nodes that need to be reset when the graph is reset */
	final ArrayList<Node> toReset = new ArrayList<>();
	
	/** A list of nodes to be included in the next level of the graph */
	final ArrayList<StepNode> nextSteps = new ArrayList<>();
	
	/** A list of level objects, created as needed */
	private final ArrayList<Level> levels = new ArrayList<>();
	
	/** The number of levels currently in the graph */
	private int size = 0;
	
	/** Whether or not the graph has leveled off */
	private boolean leveledOff = false;
	
	/**
	 * Constructs a new plan graph using
	 * {@link edu.uky.ai.planning.ss.StateSpaceProblem#steps all possible steps}
	 * for step nodes.
	 * 
	 * @param problem the problem whose steps should be used as step nodes
	 * @param persistence whether or not persistence steps should be generated
	 * @param mutexes whether or not mutexes should be calculated
	 */
	public PlanGraph(StateSpaceProblem problem, boolean persistence, boolean mutexes) {
		this.problem = problem;
		this.mutexes = mutexes;
		for(Step step : problem.steps)
			addEdgesForStep(new StepNode(this, step));
		ArrayList<Literal> literals = new ArrayList<>(this.literalMap.size());
		literals.addAll(this.literalMap.keySet());
		if(persistence) {
			for(Literal literal : literals) {
				LiteralNode literalNode = get(literal);
				StepNode persist = new StepNode(this, literal);
				stepMap.put(persist.step, persist);
				literalNode.producers.add(0, persist);
				persist.preconditions.add(literalNode);
				persist.effects.add(literalNode);
				literalNode.consumers.add(0, persist);
			}
		}
		ArrayList<LiteralNode> goals = new ArrayList<>();
		Utilities.forEachLiteral(problem.goal, literal -> {
			goals.add(getLiteralNode(literal));
		});
		this.goals = goals;
		this.levels.add(new Level(this, 0));
		if(mutexes)
			computeStaticMutexes();
	}
	
	/**
	 * Constructs a plan graph using all possible steps that could occur as the
	 * step nodes.
	 * 
	 * @param problem the problem
	 * @param persistence whether or not persistence steps should be generated
	 * @param mutexes whether or not mutexes should be calculated
	 */
	public PlanGraph(Problem problem, boolean persistence, boolean mutexes) {
		this(new StateSpaceProblem(problem), persistence, mutexes);
	}
	
	/**
	 * Connections a step node to its preconditions (and its preconditions to
	 * it) and its effects (and it effects to it).
	 * 
	 * @param stepNode the step node to and from which edges should be added
	 */
	private final void addEdgesForStep(StepNode stepNode) {
		stepMap.put(stepNode.step, stepNode);
		Utilities.forEachLiteral(stepNode.step.precondition, literal -> {
			LiteralNode literalNode = getLiteralNode(literal);
			literalNode.consumers.add(stepNode);
			stepNode.preconditions.add(literalNode);
		});
		Utilities.forEachLiteral(stepNode.step.effect, literal -> {
			LiteralNode literalNode = getLiteralNode(literal);
			stepNode.effects.add(literalNode);
			literalNode.producers.add(stepNode);
		});
	}
	
	/**
	 * Returns a literal node for a given literal object, or creates the node
	 * if one does not exist.
	 * 
	 * @param literal the literal
	 * @return the corresponding literal node
	 */
	private final LiteralNode getLiteralNode(Literal literal) {
		LiteralNode literalNode = literalMap.get(literal);
		if(literalNode == null) {
			literalNode = new LiteralNode(this, literal);
			literalMap.put(literal, literalNode);
		}
		return literalNode;
	}
	
	/**
	 * Calculates mutex relations which always exist and thus do not need to
	 * be recalculated each time to graph is reset.
	 */
	private final void computeStaticMutexes() {
		// A literal is always mutex with its negation.
		for(LiteralNode literalNode : literalMap.values()) {
			LiteralNode negation = get(literalNode.literal.negate());
			if(negation != null)
				literalNode.mutexes.add(negation, Mutexes.ALWAYS);
		}
		// Compute static mutexes for all pairs of steps.
		StepNode[] steps = stepMap.values().toArray(new StepNode[stepMap.size()]);
		for(int i=0; i<steps.length; i++) {
			for(int j=i; j<steps.length; j++) {
				if(alwaysMutex(steps[i], steps[j])) {
					steps[i].mutexes.add(steps[j], Mutexes.ALWAYS);
					steps[j].mutexes.add(steps[i], Mutexes.ALWAYS);
				}
			}
		}
	}
	
	/**
	 * Tests whether a pair of steps must always be mutex.
	 * 
	 * @param s1 the first step
	 * @param s2 the second step
	 * @return true if they are always mutex, false otherwise
	 */
	private final boolean alwaysMutex(StepNode s1, StepNode s2) {
		// Inconsistent effects: steps which undo each others' effects are always mutex.
		for(LiteralNode s1Effect : s1.effects) {
			Literal negation = s1Effect.literal.negate();
			for(LiteralNode s2Effect : s2.effects)
				if(s2Effect.literal.equals(negation))
					return true;
		}
		// Interference: steps which undo each other's preconditions are always mutex.
		if(s1 == s2)
			return false;
		if(interference(s1, s2) || interference(s2, s1))
			return true;
		return false;
	}
	
	/**
	 * Tests whether one step's effects interferes with another step's
	 * preconditions.
	 * 
	 * @param s1 the step whose effects will be checked
	 * @param s2 the step whose preconditions will be checked
	 * @return true if s1 interferes with s2
	 */
	private final boolean interference(StepNode s1, StepNode s2) {
		for(LiteralNode s1Effect : s1.effects) {
			Literal negation = s1Effect.literal.negate();
			for(LiteralNode s2Precondition : s2.preconditions)
				if(s2Precondition.literal.equals(negation))
					return true;
		}
		return false;
	}
	
	@Override
	public String toString() {
		StringWriter writer = new StringWriter();
		writer.append("Plan Graph for \"" + problem.name + "\":\n");
		for(int i=0; i<size(); i++) {
			writer.append("  Level " + i + ":\n");
			if(i > 0) {
				writer.append("    Steps:\n");
				for(StepNode node : stepMap.values())
					if(node.exists(i))
						writer.append("      " + node + "\n");
			}
			writer.append("    Literals:\n");
			for(LiteralNode node : literalMap.values())
				if(node.exists(i))
					writer.append("      " + node + "\n");
		}
		return writer.toString();
	}
	
	/**
	 * Returns a literal object's node in this graph.
	 * 
	 * @param literal the literal
	 * @return the literal's corresponding node
	 */
	public LiteralNode get(Literal literal) {
		return literalMap.get(literal);
	}
	
	/**
	 * Returns a step object's node in this graph.
	 * 
	 * @param step the step
	 * @return the step's corresponding node
	 */
	public StepNode get(Step step) {
		return stepMap.get(step);
	}
	
	/**
	 * Resets the graph to have only level 0, which will include exactly those
	 * literals which are true in the given state.
	 * 
	 * @param initial any state
	 */
	public void initialize(State initial) {
		nextSteps.clear();
		size = 1;
		for(Node node : toReset)
			node.reset();
		toReset.clear();
		for(LiteralNode node : literalMap.values())
			if(node.literal.isTrue(initial))
				node.setLevel(0);
		leveledOff = nextSteps.size() == 0;
	}
	
	/**
	 * Adds one new level to the graph.
	 */
	public void extend() {
		Level level = new Level(this, size);
		if(levels.size() == size)
			levels.add(level);
		size++;
		addStep(0);
		if(mutexes)
			level.computeMutexes();
		if(nextSteps.size() == 0)
			leveledOff = true;
	}
	
	/**
	 * Adds a step from the list of next steps to the current level.
	 * This method copies the list's contents into the system stack, clears
	 * the list, and then sets the level of each step.  This ensures that the
	 * list is empty before any step's level is set, so as to ensure that
	 * only one level of steps gets added to it at a time.
	 * 
	 * @param index the index of the step in the list of next steps to set
	 */
	private final void addStep(int index) {
		if(index == nextSteps.size())
			nextSteps.clear();
		else {
			StepNode step = nextSteps.get(index);
			addStep(index + 1);
			step.setLevel(size - 1);
		}
	}
	
	/**
	 * Returns the number of levels in this graph.
	 * 
	 * @return the number of levels
	 */
	public int size() {
		return size;
	}
	
	/**
	 * Returns a {@link Level level} object for the level of the given index.
	 * 
	 * @param number the index of the requested level
	 * @return the level
	 * @throws IndexOutOfBoundsException if the requested level does not exist in this graph
	 */
	public Level getLevel(int number) {
		if(number < 0 || number >= size)
			throw new IndexOutOfBoundsException("Level " + number + " does not exist.");
		return levels.get(number);
	}
	
	/**
	 * Tests whether all the problem's goal literals exist at the highest
	 * level of the graph.
	 * 
	 * @return true if all goals exist, false otherwise
	 */
	public boolean goalAchieved() {
		for(LiteralNode literal : goals)
			if(literal.getLevel() == -1)
				return false;
		return true;
	}
	
	/**
	 * Tests whether or not the graph has leveled off (meaning that no new
	 * literals or steps will appear if a new level is added).
	 * 
	 * @return true if the graph has leveled off, false otherwise
	 */
	public boolean hasLeveledOff() {
		return leveledOff;
	}
}
