package edu.uky.ai.planning.pg;

/**
 * The abstract superclass of {@link LiteralNode} and {@link StepNode}.
 * 
 * @author Stephen G. Ware
 */
public abstract class Node implements Comparable<Node> {

	/** The graph to which this node belongs */
	protected final PlanGraph graph;
	
	/** The index of the level at which this node first appears, or -1 if it has not yet appeared */
	protected int level = -1;
	
	/** The set of nodes with which this node is mutually exclusive */
	final Mutexes mutexes = new Mutexes(this);
	
	/** Whether or not the node needs to be reset when its plan graph is reset */
	private boolean reset = false;
	
	/**
	 * Constructs a new node object in the given plan graph.
	 * 
	 * @param graph the plan graph
	 */
	protected Node(PlanGraph graph) {
		this.graph = graph;
	}
	
	@Override
	public int compareTo(Node other) {
		return level - other.level;
	}
	
	/**
	 * Tests whether or not this node exists at a given level.
	 * 
	 * @param level the index of a level in the plan graph
	 * @return true if this node appears at the given level, false otherwise
	 */
	public boolean exists(int level) {
		return this.level != -1 && this.level <= level;
	}
	
	/**
	 * Returns the index of the earliest level at which this node appears, or
	 * -1 if this node has not yet appeared at any level.
	 * 
	 * @return this node's earliest level or -1
	 */
	public int getLevel() {
		return level;
	}
	
	/**
	 * Marks this node as having first appeared at the given level.
	 * 
	 * @param level the index of the level at which this node first appears
	 * @return true if this node's level was not previously set, false otherwise
	 */
	protected boolean setLevel(int level) {
		if(this.level == -1) {
			markForReset();
			this.level = level;
			return true;
		}
		else
			return false;
	}
	
	/**
	 * Marks this node as needing to be reset when the plan graph is reset.
	 */
	protected void markForReset() {
		if(!reset) {
			reset = true;
			graph.toReset.add(this);
		}
	}
	
	/**
	 * Tests whether or not this node is mutually exclusive with another given
	 * node at the given level.
	 * 
	 * @param node some other node in the graph
	 * @param level the index of a level in this graph
	 * @return true if these nodes are mutex at this level, false otherwise
	 * @throws IllegalArgumentException if either node does not exist at the given level
	 */
	public boolean mutex(Node node, int level) {
		return mutexes.contains(node, level);
	}
	
	/**
	 * Resets this node (called when the plan graph is being reset).
	 */
	protected void reset() {
		reset = false;
		level = -1;
		mutexes.clear();
	}
}
