package edu.uky.ai.planning.pg;

import java.util.HashMap;
import java.util.HashSet;

/**
 * A data structure for holding the set of nodes that are mutually exclusive
 * with some node and at what level those mutual exclusions exist.
 * 
 * @author Stephen G. Ware
 */
class Mutexes {
	
	/** Indicates two nodes are always mutex */
	public static final int ALWAYS = -1;

	/** The node whose mutexes are being recorded */
	private final Node node;
	
	/** The set of nodes which are always mutex with this node no matter what */
	private final HashSet<Node> staticMutexes = new HashSet<>();
	
	/** The set of nodes which may be mutex with this node depending on the state of the graph */
	private final HashMap<Node, Integer> dynamicMutexes = new HashMap<>();
	
	/**
	 * Constructs a new mutex set for a given node.
	 * 
	 * @param node the node whose mutexes are being recorded
	 */
	Mutexes(Node node) {
		this.node = node;
	}
	
	/**
	 * Indicates that the node represented by this set is mutex with another
	 * node until the given level.  The value ALWAYS indicates that this node
	 * can never not be mutex.
	 * 
	 * @param node the node with which this node is mutex
	 * @param level the highest known level at which this mutex exists
	 */
	void add(Node node, int level) {
		if(level == ALWAYS)
			staticMutexes.add(node);
		else {
			Integer end = dynamicMutexes.get(node);
			if(end == null || end < level)
				dynamicMutexes.put(node, level);
		}
	}
	
	/**
	 * Tests whether the node represented by this set is mutex with another
	 * given node at a given level.
	 * 
	 * @param node the second node
	 * @param level a level in the plan graph
	 * @return true if the nodes are mutex at that level, false otherwise
	 */
	public boolean contains(Node node, int level) {
		if(!this.node.exists(level))
			throw new IllegalArgumentException(this.node + " does not exist at level " + level + ".");
		if(!node.exists(level))
			throw new IllegalArgumentException(node + " does not exist at level " + level + ".");
		if(staticMutexes.contains(node))
			return true;
		Integer end = dynamicMutexes.get(node);
		return end != null && level <= end;
	}
	
	/**
	 * Removes all dynamic mutexes.
	 */
	public void clear() {
		dynamicMutexes.clear();
	}
}
