package edu.uky.ai.planning;

/**
 * A result contains information about how and why a search succeeded or failed.
 * 
 * @author Stephen G. Ware
 */
public class Result {

	/** The planner that performed the search */
	public final Planner<?> planner;
	
	/** The problem being solved */
	public final Problem problem;
	
	/** Whether or not the search was successful */
	public final boolean success;
	
	/** The plan, or null if the search failed */
	public final Plan solution;
	
	/** A brief, human-readable explanation of the results */
	public final String reason;
	
	/** The number of nodes visited during the search */
	public final int visited;
	
	/** The number of nodes generated during the search */
	public final int generated;
	
	/** The amount of time the search took (in milliseconds) */
	public final long time;
	
	/**
	 * Constructs a new result object.
	 * 
	 * @param planner the planner used
	 * @param problem the problem being solved
	 * @param solution the plan (possibly null)
	 * @param reason the explanation of results
	 * @param visited the number of nodes visited
	 * @param generated the number of nodes generated
	 * @param time the amount of time spent searching (in milliseconds)
	 */
	public Result(Planner<?> planner, Problem problem, Plan solution, String reason, int visited, int generated, long time) {
		this.planner = planner;
		this.problem = problem;
		if(solution == null)
			this.success = false;
		else
			this.success = problem.isSolution(solution);
		this.solution = solution;
		this.reason = reason;
		this.visited = visited;
		this.generated = generated;
		this.time = time;
	}
	
	@Override
	public String toString() {
		String str = "[" + planner + " ";
		if(success)
			str += "succeeded";
		else
			str += "failed";
		str += " on " + problem.name + " in " + problem.domain.name + "; ";
		str += visited + " visited, " + generated + " generated; ";
		str += edu.uky.ai.util.Utilities.time(time);
		if(!success)
			str += "; " + reason;
		return str + "]";
	}
}
