package edu.uky.ai.planning;

import edu.uky.ai.OperationsBudgetExceededException;
import edu.uky.ai.SearchBudget;
import edu.uky.ai.TimeBudgetExceededException;

/**
 * A planner is an algorithm which solves a planning problem by finding a plan.
 * 
 * @author Stephen G. Ware
 * @param <S> the kind of search done by this planner
 */
public abstract class Planner<S extends Search> {
	
	/** The name of the planner */
	public final String name;
	
	/**
	 * Constructs a new planner with a given name.
	 * 
	 * @param name the name of the planner
	 */
	public Planner(String name) {
		this.name = name;
	}
	
	@Override
	public String toString() {
		return name;
	}
	
	/**
	 * Returns a solution to the given problem.
	 * 
	 * @param problem the problem for which to find a solution
	 * @param maxNodes the maximum number of nodes the planner may visit during
	 * search
	 * @param maxTime the maximum number of milliseconds the planner may search
	 * @return a result object describing the result of the search
	 */
	public final Result solve(Problem problem, int maxNodes, long maxTime) {
		return solve(problem, new SearchBudget(maxNodes, maxTime));
	}
	
	/**
	 * Returns a solution to the given problem.
	 * 
	 * @param problem the problem for which to find a solution
	 * @param budget the search budget, which specifies the maximum number of
	 * nodes the planner may visit and maximum amount of time the search can
	 * take
	 * @return a result object describing the result of the search
	 */
	@SuppressWarnings("unused")
	public final Result solve(Problem problem, SearchBudget budget) {
		Plan solution = null;
		String reason = "success";
		Object[] extra = new Object[512];
		S space = makeSearch(problem, budget);
		if(space.budget != budget)
			throw new IllegalArgumentException("The provided search budget must be used.");
		long start = System.currentTimeMillis();
		try {
			solution = space.solve();
			if(solution == null)
				reason = "no solution exists";
			else if(!problem.isSolution(solution))
				reason = "invalid solution: " + solution;
		}
		catch(OperationsBudgetExceededException ex) {
			reason = "node limit reached";
		}
		catch(TimeBudgetExceededException ex) {
			reason = "time limit reached";
		}
		catch(OutOfMemoryError ex) {
			extra = null;
			System.gc();
			reason = "out of memory";
		}
		long time = System.currentTimeMillis() - start;
		return new Result(this, problem, solution, reason, space.countVisited(), space.countGenerated(), time);
	}
	
	/**
	 * Given some problem to be solved, and search budget in which to solve it,
	 * this method constructs the appropriate kind of {@link Search} to solve
	 * the problem based on this planner's technique.
	 * 
	 * @param problem the problem to be solved
	 * @param budget the search budget, which specifies the maximum number of
	 * nodes the planner may visit and maximum amount of time the search can
	 * take
	 * @return a search object for solving this problem
	 */
	protected abstract S makeSearch(Problem problem, SearchBudget budget);
}
