package edu.uky.ai.planning;

import edu.uky.ai.logic.Conjunction;
import edu.uky.ai.logic.Disjunction;
import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.Substitution;
import edu.uky.ai.logic.Variable;
import edu.uky.ai.util.ImmutableArray;

/**
 * An operator is an action template that describes one way to change the world in terms
 * of its precondition (what must be true before the action can be taken) and effect
 * (what becomes true after the action is taken).
 * 
 * @author Stephen G. Ware
 */
public class Operator {

	/** The name of the action */
	public final String name;
	
	/** The parameters that provide the specific details for the action */
	public final ImmutableArray<Variable> parameters;
	
	/** What must be true before the action can be taken */
	public final Proposition precondition;
	
	/** What becomes true after the action is taken */
	public final Proposition effect;
	
	/**
	 * Constructs a new action template.
	 * 
	 * @param name the name of the action
	 * @param parameters the parameters that provide specific detail
	 * @param precondition what must be true before
	 * @param effect what becomes true after
	 */
	public Operator(String name, ImmutableArray<Variable> parameters, Proposition precondition, Proposition effect) {
		if(!isDeterministic(effect))
			throw new IllegalArgumentException("Effect nondeterministic: " + effect.toDNF());
		this.name = name;
		this.parameters = parameters;
		this.precondition = precondition;
		this.effect = effect;
	}
	
	/**
	 * Constructs a new action template.
	 * 
	 * @param name the name of the action
	 * @param parameters the parameters that provide specific detail
	 * @param precondition what must be true before
	 * @param effect what becomes true after
	 */
	public Operator(String name, Variable[] parameters, Proposition precondition, Proposition effect) {
		this(name, new ImmutableArray<>(parameters), precondition, effect);
	}
	
	/**
	 * Checks if an effect is deterministic (i.e. results in exactly one
	 * possible next state).
	 * 
	 * @param proposition the proposition to test
	 * @return true of the proposition is deterministic, false otherwise
	 */
	private static final boolean isDeterministic(Proposition proposition) {
		proposition = proposition.toDNF();
		if(proposition instanceof Literal || proposition instanceof Conjunction)
			return true;
		else {
			if(((Disjunction) proposition).arguments.size() == 1)
				return true;
			else
				return false;
		}
	}
	
	/**
	 * Creates a ground step (i.e. a specific action) from this action
	 * template.
	 * 
	 * @param substitution provides bindings for each of the operator's parameters
	 * @return a step
	 */
	public Step makeStep(Substitution substitution) {
		String name = "(" + this.name;
		for(Variable parameter : parameters)
			name += " " + parameter.substitute(substitution);
		name += ")";
		return new Step(name, precondition.substitute(substitution), effect.substitute(substitution));
	}
	
	@Override
	public String toString() {
		String str = "(" + name;
		for(Variable parameter : parameters)
			str += " "  + parameter;
		return str + ")";
	}
}
