package edu.uky.ai.logic.io;

import edu.uky.ai.io.*;
import edu.uky.ai.logic.Atom;
import edu.uky.ai.logic.Constant;
import edu.uky.ai.logic.Predication;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.Term;
import edu.uky.ai.logic.Variable;

/**
 * A parser for predicate logic, which expects all {@link Atom}s to be
 * {@link Predication}s.
 * 
 * @author Stephen G. Ware
 */
public class PredicateLogicParser extends LogicParser {

	/**
	 * Constructs a new parser with rules for parsing predicate logic formulas.
	 */
	public PredicateLogicParser() {
		setParser(Atom.class, ATOM_PARSER);
		setParser(Predication.class, PREDICATION_PARSER);
		setParser(Term.class, TERM_PARSER);
		setParser(Constant.class, CONSTANT_PARSER);
		setParser(Variable.class, VARIABLE_PARSER);
	}
	
	/** Parses atoms */
	private static final ObjectParser<Atom> ATOM_PARSER = new ObjectParser<Atom>() {

		@Override
		public Atom parse(Node node, Parser parser) {
			if(node.isSymbol(Proposition.TRUE.toString()))
				return Proposition.TRUE;
			else if(node.isSymbol(Proposition.FALSE.toString()))
				return Proposition.FALSE;
			else
				return parser.parse(node, Predication.class);
		}
	};
	
	/** Parses predications */
	private static final ObjectParser<Predication> PREDICATION_PARSER = new ObjectParser<Predication>() {

		@Override
		public Predication parse(Node node, Parser parser) {
			node = node.asList().requireFirst();
			String predicate = node.asSymbol().value;
			return new Predication(predicate, parseEachAs(parser, node.next, Term.class));
		}
	};
	
	/** Parses terms */
	private static final ObjectParser<Term> TERM_PARSER = new ObjectParser<Term>() {

		@Override
		public Term parse(Node node, Parser parser) {
			return parser.parse(node, Constant.class, Variable.class);
		}
	};
	
	/** Parses constants */
	private static final ObjectParser<Constant> CONSTANT_PARSER = new ObjectParser<Constant>() {

		@Override
		public Constant parse(Node node, Parser parser) {
			if(node.isSymbol() && !node.asSymbol().value.startsWith("?"))
				return parser.requireDefined(node.asSymbol().value, Constant.class);
			else
				return null;
		}
	};
	
	/** Parses variables */
	private static final ObjectParser<Variable> VARIABLE_PARSER = new ObjectParser<Variable>() {

		@Override
		public Variable parse(Node node, Parser parser) {
			if(node.isSymbol() && node.asSymbol().value.startsWith("?"))
				return parser.requireDefined(node.asSymbol().value.substring(1), Variable.class);
			else
				return null;
		}
	};
}
