package edu.uky.ai.logic.io;

import java.lang.reflect.Array;
import java.util.ArrayList;

import edu.uky.ai.io.*;
import edu.uky.ai.logic.Atom;
import edu.uky.ai.logic.Conjunction;
import edu.uky.ai.logic.Disjunction;
import edu.uky.ai.logic.Implication;
import edu.uky.ai.logic.Literal;
import edu.uky.ai.logic.NegatedAtom;
import edu.uky.ai.logic.Negation;
import edu.uky.ai.logic.Proposition;

/**
 * Contains common parsing rules that apply to many kinds of logic.
 * 
 * @author Stephen G. Ware
 */
public abstract class LogicParser extends Parser {

	/**
	 * Constructs a new parser.
	 */
	public LogicParser() {
		setParser(Proposition.class, PROPOSITION_PARSER);
		setParser(Negation.class, NEGATION_PARSER);
		setParser(Conjunction.class, CONJUNCTION_PARSER);
		setParser(Disjunction.class, DISJUNCTION_PARSER);
		setParser(Implication.class, IMPLICATION_PARSER);
		setParser(Literal.class, LITERAL_PARSER);
		setParser(NegatedAtom.class, NEGATED_ATOM_PARSER);
		setParser(Atom.class, ATOM_PARSER);
	}
	
	/** Parses expressions */
	private static final ObjectParser<Proposition> PROPOSITION_PARSER = new ObjectParser<Proposition>() {

		@Override
		public Proposition parse(Node node, Parser parser) {
			return parser.parse(node, Negation.class, Conjunction.class, Disjunction.class, Implication.class, Literal.class);
		}
	};
	
	/** Parses negations */
	private static final ObjectParser<Negation> NEGATION_PARSER = new ObjectParser<Negation>() {

		@Override
		public Negation parse(Node node, Parser parser) {
			if(node.isList(1, -1) && node.asList().first.isSymbol(Negation.NEGATION_PREDICATE)) {
				Proposition argument = parser.parse(node.asList(2, 2).requireFirst().next, Proposition.class);
				if(argument instanceof Atom)
					return new NegatedAtom((Atom) argument);
				else
					return new Negation(argument);
			}
			return null;
		}
	};
	
	/** Parses conjunctions */
	private static final ObjectParser<Conjunction> CONJUNCTION_PARSER = new ObjectParser<Conjunction>() {

		@Override
		public Conjunction parse(Node node, Parser parser) {
			if(node.isList(1, -1) && node.asList(1, -1).first.isSymbol(Conjunction.CONJUNCTION_PREDICATE))
				return new Conjunction(parseEachAs(parser, node.asList(1, -1).requireFirst().next, Proposition.class));
			else
				return null;
			
		}
	};
	
	/** Parses disjunctions */
	private static final ObjectParser<Disjunction> DISJUNCTION_PARSER = new ObjectParser<Disjunction>() {

		@Override
		public Disjunction parse(Node node, Parser parser) {
			if(node.isList(1, -1) && node.asList().first.isSymbol(Disjunction.DISJUNCTION_PREDICATE))
				return new Disjunction(parseEachAs(parser, node.asList(1, -1).requireFirst().next, Proposition.class));
			else
				return null;
		}
	};
	
	/** Parses implications */
	private static final ObjectParser<Implication> IMPLICATION_PARSER = new ObjectParser<Implication>() {

		@Override
		public Implication parse(Node node, Parser parser) {
			if(node.isList(1, -1) && node.asList().first.isSymbol(Implication.IMPLICATION_PREDICATE)) {
				Proposition antecedent = parser.parse(node.asList(3, 3).first.requireNext(), Proposition.class);
				Proposition consequent = parser.parse(node.asList().first.requireNext().requireNext(), Proposition.class);
				return new Implication(antecedent, consequent);
			}
			else
				return null;
		}
	};
	
	/** Parses literals */
	private static final ObjectParser<Literal> LITERAL_PARSER = new ObjectParser<Literal>() {

		@Override
		public Literal parse(Node node, Parser parser) {
			return parser.parse(node, NegatedAtom.class, Atom.class);
		}
	};
	
	/** Parses negated literals */
	private static final ObjectParser<NegatedAtom> NEGATED_ATOM_PARSER = new ObjectParser<NegatedAtom>() {

		@Override
		public NegatedAtom parse(Node node, Parser parser) {
			if(node.isList(1, -1) && node.asList().first.isSymbol(Negation.NEGATION_PREDICATE))
				return new NegatedAtom(parser.parse(node.asList(2, 2).requireFirst().next, Atom.class));
			else
				return null;
		}
	};
	
	/** Parses atoms */
	private static final ObjectParser<Atom> ATOM_PARSER = new ObjectParser<Atom>() {

		@Override
		public Atom parse(Node node, Parser parser) {
			if(node.isSymbol(Proposition.TRUE.toString()))
				return Proposition.TRUE;
			else if(node.isSymbol(Proposition.FALSE.toString()))
				return Proposition.FALSE;
			else
				return null;
		}
	};
	
	@SuppressWarnings("unchecked")
	static final <T> T[] parseEachAs(Parser parser, Node first, Class<T> type) {
		ArrayList<T> results = new ArrayList<>();
		while(first != null) {
			results.add(parser.parse(first, type));
			first = first.next;
		}
		return results.toArray((T[]) Array.newInstance(type, results.size()));
	}
}
