package edu.uky.ai.logic;

import java.util.ArrayList;
import java.util.LinkedHashSet;

import edu.uky.ai.util.ImmutableArray;

/**
 * A conjunction, or logical AND statement, is true just when all of its
 * arguments are true.
 * 
 * @author Stephen G. Ware
 */
public class Conjunction extends BooleanProposition {
	
	/** The word used to express a conjunction */
	public static final String CONJUNCTION_PREDICATE = "and";

	/**
	 * Constructs a conjunction from a group of arguments.
	 * 
	 * @param arguments the arguments
	 */
	public Conjunction(ImmutableArray<Proposition> arguments) {
		super(arguments);
	}

	/**
	 * Constructs a conjunction from a group of arguments.
	 * 
	 * @param arguments the arguments
	 */
	public Conjunction(Proposition...arguments) {
		this(new ImmutableArray<>(arguments));
	}
	
	/**
	 * Constructs a conjunction from a group of arguments.
	 * 
	 * @param arguments the arguments
	 */
	public Conjunction(Iterable<? extends Proposition> arguments) {
		this(edu.uky.ai.util.Utilities.toArray(arguments, Proposition.class));
	}
	
	@Override
	public boolean equals(Object other) {
		return other instanceof Conjunction && argumentsEqual(other);
	}
	
	@Override
	public int hashCode() {
		return Utilities.hashCode(CONJUNCTION_PREDICATE, arguments);
	}
	
	@Override
	public String toString() {
		return Utilities.toString(CONJUNCTION_PREDICATE, arguments);
	}

	@Override
	public Proposition substitute(Substitution substitution) {
		ImmutableArray<Proposition> arguments = substituteArguments(substitution);
		if(arguments == this.arguments)
			return (Proposition) substitution.get(this);
		else
			return (Proposition) substitution.get(new Conjunction(arguments));
	}
	
	@Override
	public Bindings unify(Formula other, Bindings bindings) {
		if(other instanceof Conjunction)
			return unifyArguments(other, bindings);
		else
			return null;
	}

	@Override
	public boolean isTrue(State state) {
		for(Proposition argument : arguments)
			if(!argument.isTrue(state))
				return false;
		return true;
	}

	@Override
	public void makeTrue(MutableState state) {
		for(Proposition argument : arguments)
			argument.makeTrue(state);
	}

	@Override
	public Proposition simplify() {
		LinkedHashSet<Proposition> arguments = new LinkedHashSet<>();
		if(!collect(this.arguments, arguments))
			return FALSE;
		else if(arguments.size() == 0)
			return TRUE;
		else if(arguments.size() == 1)
			return arguments.iterator().next();
		else
			return new Conjunction(arguments);
	}
	
	private static final boolean collect(ImmutableArray<Proposition> arguments, LinkedHashSet<Proposition> set) {
		for(Proposition argument : arguments) {
			argument = argument.simplify();
			if(argument == TRUE)
				continue;
			else if(argument == FALSE)
				return false;
			else if(argument instanceof Conjunction) {
				if(!collect(((Conjunction) argument).arguments, set))
					return false;
			}
			else
				set.add(argument);
		}
		return true;
	}

	@Override
	public Disjunction negate() {
		return new Disjunction(negateArguments());
	}

	@Override
	public Proposition toCNF() {
		ArrayList<Proposition> clauses = new ArrayList<>();
		for(Proposition argument : arguments) {
			argument = argument.toCNF();
			if(argument instanceof Literal || argument instanceof Disjunction)
				clauses.add(argument);
			else
				for(Proposition clause : ((Conjunction) argument).arguments)
					clauses.add(clause);
		}
		return new Conjunction(clauses);
	}

	@Override
	public Proposition toDNF() {
		return Utilities.recombine(this, Utilities.DNF);
	}
}
