package edu.uky.ai.logic;

import edu.uky.ai.util.ImmutableArray;

/**
 * The parent class of all Boolean expressions, whose truth value is defined by
 * combining other expressions.
 * 
 * @author Stephen G. Ware
 */
public abstract class BooleanProposition implements Proposition {
	
	/** The other propositions that are part of this Boolean proposition */
	public final ImmutableArray<Proposition> arguments;
	
	/**
	 * Constructs a new Boolean proposition with the given arguments.
	 * 
	 * @param arguments the other propositions that are part of this
	 * proposition
	 */
	public BooleanProposition(ImmutableArray<Proposition> arguments) {
		this.arguments = arguments;
	}

	@Override
	public boolean isGround() {
		for(int i=0; i<arguments.size(); i++)
			if(!arguments.get(i).isGround())
				return false;
		return true;
	}
	
	/**
	 * Tests whether this formula's arguments are equal to the arguments of
	 * another Boolean proposition.
	 * 
	 * @param other must be an instance of {@link BooleanProposition}
	 * @return true if both sets of arguments are equal, false otherwise
	 */
	protected boolean argumentsEqual(Object other) {
		return arguments.equals(((BooleanProposition) other).arguments);
	}
	
	/**
	 * Returns the arguments of this formula after applying the given
	 * substitution.
	 * 
	 * @param substitution the substitution to apply
	 * @return this formula's arguments after applying the substitution
	 */
	protected ImmutableArray<Proposition> substituteArguments(Substitution substitution) {
		return Utilities.substitute(arguments, substitution);
	}
	
	/**
	 * Unifies this formula's arguments with the arguments of another Boolean
	 * proposition. See {@link Formula#unify(Formula, Bindings)}.
	 * 
	 * @param other must be an instance of {@link BooleanProposition}
	 * @param bindings an existing set of bindings that will be added to
	 * @return the bindings that would make both formulas the same, or null if
	 * so such bindings exist
	 */
	protected Bindings unifyArguments(Formula other, Bindings bindings) {
		BooleanProposition otherBoolean = (BooleanProposition) other;
		if(arguments.size() != otherBoolean.arguments.size())
			return null;
		for(int i=0; i<arguments.size() && bindings != null; i++)
			bindings = arguments.get(i).unify(otherBoolean.arguments.get(i), bindings);
		return bindings;
	}
	
	/**
	 * Returns the arguments of this formula after negating each of them.
	 * See {@link Proposition#negate()}.
	 * 
	 * @return the negated arguments
	 */
	protected ImmutableArray<Proposition> negateArguments() {
		Proposition[] arguments = new Proposition[this.arguments.size()];
		for(int i=0; i<arguments.length; i++)
			arguments[i] = this.arguments.get(i).negate();
		return new ImmutableArray<>(arguments);
	}
}
