package edu.uky.ai.util;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.function.Consumer;

import edu.uky.ai.Settings;

/**
 * An array whose values cannot be modified.
 * 
 * @author Stephen G. Ware
 * @param <E> the type of object kept in the array
 */
public class ImmutableArray<E> implements Serializable, Iterable<E> {
	
	/** Uniquely identifies the version number for serialization and deserialization */
	private static final long serialVersionUID = Settings.VERSION_UID;
	
	/** The array being protected by this class */
	private final E[] array;
	
	/**
	 * Constructs a new immutable array which reflects the given array.
	 * 
	 * @param array the array to mirror
	 */
	public ImmutableArray(E[] array) {
		this.array = array;
	}
	
	/**
	 * Creates an immutable array that contains the elements of a collection.
	 * 
	 * @param collection the collection to mirror
	 * @param type the type of object kept in the array
	 */
	@SuppressWarnings("unchecked")
	public ImmutableArray(Collection<E> collection, Class<E> type) {
		this.array = collection.toArray((E[]) Array.newInstance(type, collection.size()));
	}
	
	@Override
	public boolean equals(Object other) {
		return other instanceof ImmutableArray<?> && Arrays.equals(array, ((ImmutableArray<?>) other).array);
	}
	
	@Override
	public int hashCode() {
		return Arrays.hashCode(array);
	}
	
	@Override
	public String toString() {
		return Arrays.toString(array);
	}
	
	/**
	 * Returns the number of elements in the array.
	 * 
	 * @return the number of elements
	 */
	public int size() {
		return array.length;
	}
	
	/**
	 * Returns the element at a given index.
	 * 
	 * @param index the index in the array
	 * @return the element at that index
	 */
	public final E get(int index) {
		return array[index];
	}
	
	/**
	 * Returns the array's component type.
	 * 
	 * @return the component type
	 */
	@SuppressWarnings("unchecked")
	public Class<E> getComponentType() {
		return (Class<E>) array.getClass().getComponentType();
	}
	
	/**
	 * Checks if the array contains a given element.
	 * 
	 * @param element the element to search for
	 * @return true if the array contains an object equal to the given object, false otherwise
	 */
	public boolean contains(Object element) {
		return indexOf(element) != -1;
	}
	
	/**
	 * Returns the index of the first object that is equal to a given object.
	 * 
	 * @param element the element to search for
	 * @return the index of that object in the array, or -1 if no such object exists
	 */
	public int indexOf(Object element) {
		for(int i=0; i<array.length; i++)
			if(array[i].equals(element))
				return i;
		return -1;
	}
	
	@Override
	public Iterator<E> iterator() {
		return new ArrayIterator<E>(array);
	}
	
	@Override
	public void forEach(Consumer<? super E> consumer) {
		for(int i=0; i<array.length; i++)
			consumer.accept(array[i]);
	}
}
