package edu.uky.ai.sat;

import edu.uky.ai.OperationsBudgetExceededException;
import edu.uky.ai.SearchBudget;
import edu.uky.ai.TimeBudgetExceededException;

/**
 * The parent class of all satisfiability solving algorithms.
 * 
 * @author Stephen G. Ware
 */
public abstract class Solver {
	
	/** The solver's name */
	public final String name;
	
	/**
	 * Constructs a new satisfiability solver with the given name.
	 * 
	 * @param name the name
	 */
	public Solver(String name) {
		this.name = name;
	}
	
	@Override
	public int hashCode() {
		return name.hashCode();
	}
	
	@Override
	public String toString() {
		return name;
	}
	
	/**
	 * Solves the given problem, limited to the given number of operations and
	 * amount of time.
	 * 
	 * @param problem the satisfiability problem to solve
	 * @param maxOperations the maximum number of operations that may be
	 * performed while searching for a solution
	 * @param maxTime the maximum number of milliseconds that the search may
	 * take
	 * @return the {@link Result} of the search
	 */
	public final Result solve(Problem problem, int maxOperations, long maxTime) {
		return solve(problem, new SearchBudget(maxOperations, maxTime));
	}
	
	/**
	 * Solves the given problem, limited by the given search budget.
	 * 
	 * @param problem the satisfiability problem to solve
	 * @param budget the search budget, which constrains how many operations
	 * may be performed and how long the search may take
	 * @return the {@link Result} of the search
	 */
	@SuppressWarnings("unused")
	public final Result solve(Problem problem, SearchBudget budget) {
		Object[] extra = new Object[512];
		String reason;
		Assignment assignment = new Assignment(problem, budget);
		int operations = budget.getOperations();
		long start = System.currentTimeMillis();
		try {
			if(solve(assignment) && assignment.getValue() == Value.TRUE)
				reason = "success";
			else
				reason = "failure";
		}
		catch(OperationsBudgetExceededException ex) {
			reason = "operations limit exceeded";
		}
		catch(TimeBudgetExceededException ex) {
			reason = "time limit exceeded";
		}
		catch(OutOfMemoryError ex) {
			extra = null;
			System.gc();
			reason = "out of memory";
		}
		catch(Exception ex) {
			reason = ex.getMessage();
		}
		return new Result(this, problem, assignment, reason, budget.getOperations() - operations, System.currentTimeMillis() - start);
	}

	/**
	 * Finds a set of {@link Value} assignments to {@link Variable}s that
	 * satisfies the proposition given in the {@link Problem} the given
	 * assignment is associated with.
	 * 
	 * @param assignment the assignment whose variable values will be set
	 * @return true if a satisfying assignment was found, false otherwise
	 */
	public abstract boolean solve(Assignment assignment);
}
