package edu.uky.ai.sat;

import edu.uky.ai.logic.Disjunction;
import edu.uky.ai.util.ImmutableArray;

/**
 * Represents one of the disjunctive clauses that make up the logical formula
 * in a satisfiability {@link Problem}. It is true when at least one of its
 * {@link Literal}s is true.
 * 
 * @author Stephen G. Ware
 */
public class Clause {

	/** The problem this clause is a part of */
	public final Problem problem;
	
	/** The number of the clause */
	public final int index;
	
	/** The proposition this clause was created from */
	public final Disjunction proposition;
	
	/** The {@link Literal}s that make up this clause */
	public final ImmutableArray<Literal> literals;
	
	private int hashCode;
	
	Clause(Problem problem, int index, Disjunction proposition, Literal[] literals) {
		this.proposition = proposition;
		this.problem = problem;
		this.index = index;
		this.literals = new ImmutableArray<>(literals);
		this.hashCode = proposition.hashCode();
	}
	
	@Override
	public int hashCode() {
		return hashCode;
	}
	
	@Override
	public String toString() {
		return proposition.toString();
	}
	
	/**
	 * Returns a string representation of the clause after substituting any
	 * variables whose values are known and simplifying the formula.
	 * 
	 * @param assignment an assignment that defines which variables have known
	 * values
	 * @return a string representation of the clause
	 */
	public String toString(Assignment assignment) {
		return proposition.substitute(assignment).simplify().toString();
	}
}
