package edu.uky.ai.sat;

import java.util.ArrayList;

import edu.uky.ai.SearchBudget;
import edu.uky.ai.logic.Atom;
import edu.uky.ai.logic.Conjunction;
import edu.uky.ai.logic.Formula;
import edu.uky.ai.logic.Proposition;
import edu.uky.ai.logic.Substitution;

/**
 * Represents a (full or partial) solution to a satisfiability {@link Problem}.
 * Initially, all variables are set to unknown.
 * 
 * @author Stephen G. Ware
 */
public class Assignment implements Cloneable, Substitution {

	/** The problem this assignment is associated with */
	public final Problem problem;
	
	/** 
	 * The search budget, which constrains how many time variable values may
	 * be set and how much time may be spent finding a solution
	 */
	public final SearchBudget budget;
	
	final Value[] variableValues;
	final int[] trueLiterals;
	final int[] falseLiterals;
	int trueClauses;
	int falseClauses;
	
	/**
	 * Creates a new assignment that is a clone of the given assignment.
	 * 
	 * @param toClone the assignment to copy
	 */
	protected Assignment(Assignment toClone) {
		this.problem = toClone.problem;
		this.budget = toClone.budget;
		this.variableValues = toClone.variableValues.clone();
		this.trueLiterals = toClone.trueLiterals.clone();
		this.falseLiterals = toClone.falseLiterals.clone();
		this.trueClauses = toClone.trueClauses;
		this.falseClauses = toClone.falseClauses;
	}
	
	/**
	 * Creates a new assignment for the given problem using the given search
	 * budget.
	 * 
	 * @param problem the problem
	 * @param budget the search budget
	 */
	Assignment(Problem problem, SearchBudget budget) {
		this.problem = problem;
		this.budget = budget;
		this.variableValues = new Value[problem.variables.size()];
		for(int i=0; i<this.variableValues.length; i++)
			this.variableValues[i] = Value.UNKNOWN;
		this.trueLiterals = new int[problem.clauses.size()];
		this.falseLiterals = new int[problem.clauses.size()];
		this.trueClauses = 0;
		this.falseClauses = 0;
		for(Clause clause : problem.clauses) {
			for(Literal literal : clause.literals) {
				if(getValue(literal) == Value.TRUE)
					trueLiterals[clause.index]++;
				else if(getValue(literal) == Value.FALSE)
					falseLiterals[clause.index]++;
			}
			if(getValue(clause) == Value.TRUE)
				trueClauses++;
			else if(getValue(clause) == Value.FALSE)
				falseClauses++;
		}
	}
	
	@Override
	public String toString() {
		String string = "{";
		for(Variable variable : problem.variables) {
			Value value = getValue(variable);
			if(value != Value.UNKNOWN)
				string += " " + variable + "=" + (value == Value.TRUE ? "T" : "F");
		}
		return string + " }";
	}
	
	@Override
	public Assignment clone() {
		return new Assignment(this);
	}
	
	@Override
	public Formula get(Formula original) {
		if(problem.variablesByAtom.containsKey(original)) {
			Value value = getValue((Variable) problem.variablesByAtom.get(original));
			if(value == Value.TRUE)
				return Proposition.TRUE;
			else if(value == Value.FALSE)
				return Proposition.FALSE;
		}
		return original;
	}
	
	/**
	 * Returns a logical proposition representing this assignment's current
	 * state (a conjunction of all variables currently set to true).
	 * 
	 * @return the proposition
	 */
	public Proposition toProposition() {
		ArrayList<Atom> positive = new ArrayList<>();
		for(Variable variable : problem.variables)
			if(getValue(variable) == Value.TRUE)
				positive.add(variable.proposition);
		if(positive.size() == 0)
			return Proposition.TRUE;
		else if(positive.size() == 1)
			return positive.get(0);
		else
			return new Conjunction(positive);
	}
	
	/**
	 * Returns the current {@link Value} of a variable in the formula.
	 * This operation runs in constant time.
	 * 
	 * @param variable the variable whose value is to be checked
	 * @return the variable's current value
	 */
	public Value getValue(Variable variable) {
		checkProblem(variable, variable.problem);
		return variableValues[variable.index];
	}
	
	/**
	 * Returns the number of literals that are currently true in a given
	 * clause. This operation runs in constant time.
	 * 
	 * @param clause the clause for which the true literals should be counted
	 * @return the number of true literals in the clause
	 */
	public int countTrueLiterals(Clause clause) {
		checkProblem(clause, clause.problem);
		return trueLiterals[clause.index];
	}
	
	/**
	 * Returns the number of literals that are currently false in a given
	 * clause. This operation runs in constant time.
	 * 
	 * @param clause the clause for which the false literals should be counted
	 * @return the number of false literals in the clause
	 */
	public int countFalseLiterals(Clause clause) {
		checkProblem(clause, clause.problem);
		return falseLiterals[clause.index];
	}
	
	/**
	 * Returns the number of literals whose current values is not known to be
	 * true or false in the given clause. This operation runs in constant time.
	 * 
	 * @param clause the clause for which the unknown literals should be
	 * counted
	 * @return the number of unknown literals
	 */
	public int countUnknownLiterals(Clause clause) {
		checkProblem(clause, clause.problem);
		return clause.literals.size() - (countTrueLiterals(clause) + countFalseLiterals(clause));
	}
	
	/**
	 * Returns the number of clauses that are currently true in the problem
	 * this assignment is associated with. This operation runs in constant
	 * time. The number of true clauses is often used as a simple measure of
	 * the quality of an assignment.
	 * 
	 * @return the number of true clauses
	 */
	public int countTrueClauses() {
		return trueClauses;
	}
	
	/**
	 * Returns the number of clauses that are currently false in the problem
	 * this assignment is associated with. This operation runs in constant
	 * time.
	 * 
	 * @return the number of false clauses
	 */
	public int countFalseClauses() {
		return falseClauses;
	}
	
	/**
	 * Returns the number of clauses whose value is not currently known in the
	 * problem this assignment is associated with. This operation runs in
	 * constant time.
	 * 
	 * @return the number of unknown clauses
	 */
	public int countUnknownClauses() {
		return problem.clauses.size() - (countTrueClauses() + countFalseClauses());
	}
	
	/**
	 * Returns the {@link Value} of a given literal. This operation runs in
	 * constant time.
	 * 
	 * @param literal the literal whose value is to be checked
	 * @return the value
	 */
	public Value getValue(Literal literal) {
		checkProblem(literal, literal.clause.problem);
		if(literal.valence)
			return getValue(literal.variable);
		else
			return getValue(literal.variable).negate();
	}
	
	/**
	 * Returns the {@link Value} of a given clause. This operation runs in
	 * constant time.
	 * 
	 * @param clause the clause whose value is to be checked
	 * @return the value
	 */
	public Value getValue(Clause clause) {
		checkProblem(clause, clause.problem);
		if(countTrueLiterals(clause) > 0)
			return Value.TRUE;
		else if(countFalseLiterals(clause) == clause.literals.size())
			return Value.FALSE;
		else
			return Value.UNKNOWN;
	}
	
	/**
	 * Returns the {@link Value} of the logical formula for the problem this
	 * assignment is associated with. This operation runs in constant time.
	 * 
	 * @return the value
	 */
	public Value getValue() {
		if(countTrueClauses() == problem.clauses.size())
			return Value.TRUE;
		else if(countFalseClauses() > 0)
			return Value.FALSE;
		else
			return Value.UNKNOWN;
	}
	
	/**
	 * Sets the {@link Value} of the given variable. This operation runs in
	 * linear time equal to the number of literals the variable appears. Each
	 * time this operation is performed, the search budget's operations is
	 * reduced by one.
	 * 
	 * @param variable the variable whose value will be set
	 * @param value the value the variable will be set to
	 */
	public void setValue(Variable variable, Value value) {
		checkProblem(variable, variable.problem);
		budget.incrementOperations();
		budget.checkTime();
		Value oldVariableValue = getValue(variable);
		if(oldVariableValue != value)
			setVariable(variable, oldVariableValue, value);
	}
	
	private final void setVariable(Variable variable, Value oldVariableValue, Value newVariableValue) {
		for(Literal literal : variable.literals) {
			variableValues[variable.index] = oldVariableValue;
			Value oldLiteralValue = getValue(literal);
			variableValues[variable.index] = newVariableValue;
			Value newLiteralValue = getValue(literal);
			if(oldLiteralValue != newLiteralValue)
				setLiteral(literal, oldLiteralValue, newLiteralValue);
		}
	}
	
	private final void setLiteral(Literal literal, Value oldLiteralValue, Value newLiteralValue) {
		Value oldClauseValue = getValue(literal.clause);
		if(oldLiteralValue == Value.TRUE)
			trueLiterals[literal.clause.index]--;
		else if(oldLiteralValue == Value.FALSE)
			falseLiterals[literal.clause.index]--;
		if(newLiteralValue == Value.TRUE)
			trueLiterals[literal.clause.index]++;
		else if(newLiteralValue == Value.FALSE)
			falseLiterals[literal.clause.index]++;
		Value newClauseValue = getValue(literal.clause);
		if(oldClauseValue != newClauseValue)
			setClause(literal.clause, oldClauseValue, newClauseValue);
	}
	
	private final void setClause(Clause clause, Value oldClauseValue, Value newClauseValue) {
		if(oldClauseValue == Value.TRUE)
			trueClauses--;
		else if(oldClauseValue == Value.FALSE)
			falseClauses--;
		if(newClauseValue == Value.TRUE)
			trueClauses++;
		else if(newClauseValue == Value.FALSE)
			falseClauses++;
	}
	
	final void checkProblem(Object object, Problem problem) {
		if(problem != this.problem)
			throw new IllegalArgumentException(object + " is from a different problem than the one this assignment is for.");
	}
}
