package edu.uky.ai.logic;

/**
 * A proposition is any logical formula with a truth value.
 * 
 * @author Stephen G. Ware
 */
public interface Proposition extends Formula {

	@Override
	public Proposition substitute(Substitution substitution);
	
	/**
	 * Tests whether this proposition is true in the given state.
	 * 
	 * @param state the state in which to test this proposition
	 * @return true if the proposition is true, false otherwise
	 */
	public boolean isTrue(State state);
	
	/**
	 * Modifies the given state object to make this proposition true. Note that
	 * some propositions (such as {@link Disjunction}s) are non-deterministic,
	 * meaning there are multiple ways to make them true, and thus the state
	 * that would result from this operation is not clearly defined.
	 * 
	 * @param state the state to modify
	 * @throws IllegalStateException if there is more than one way to make this
	 * proposition true
	 */
	public void makeTrue(MutableState state);
	
	/**
	 * Returns an equivalent proposition which may be simpler.
	 * 
	 * @return the simplified proposition
	 */
	public Proposition simplify();
	
	/**
	 * Returns a proposition whose truth values are opposite of this
	 * proposition (i.e. true when this one is false, false when this one is
	 * true).
	 * 
	 * @return a proposition with opposite truth values
	 */
	public Proposition negate();
	
	/**
	 * Converts this proposition to Conjunctive Normal Form (CNF). There are
	 * three kinds of formulas that are considered to be in CNF:
	 * <ol>
	 * <li>A {@link Literal} by itself</li>
	 * <li>A {@link Disjunction} of literals, called a clause</li>
	 * <li>A {@link Conjunction} of clauses</li>
	 * </ol>
	 * 
	 * @return an equivalent proposition in CNF
	 */
	public Proposition toCNF();
	
	/**
	 * Converts this proposition to Disjunctive Normal Form (DNF). There are
	 * three kinds of formulas that are considered to be in DNF:
	 * <ol>
	 * <li>A {@link Literal} by itself</li>
	 * <li>A {@link Conjunction} of literals, called a clause</li>
	 * <li>A {@link Disjunction} of clauses</li>
	 * </ol>
	 * 
	 * @return an equivalent proposition in DNF
	 */
	public Proposition toDNF();
	
	/** A proposition which is always trivially true */
	public final Atom TRUE = new Atom() {
		
		@Override
		public String toString() {
			return "true";
		}
		
		@Override
		public boolean isGround() {
			return true;
		}
		
		@Override
		public Atom substitute(Substitution substitution) {
			return this;
		}

		@Override
		public Bindings unify(Formula other, Bindings bindings) {
			if(this.equals(other, bindings))
				return bindings;
			else
				return null;
		}

		@Override
		public boolean isTrue(State state) {
			return true;
		}

		@Override
		public void makeTrue(MutableState state) {
			return;
		}

		@Override
		public Atom negate() {
			return FALSE;
		}
	};
	
	/** A proposition which is always trivially false */
	public final Atom FALSE = new Atom() {

		@Override
		public String toString() {
			return "false";
		}
		
		@Override
		public boolean isGround() {
			return true;
		}

		@Override
		public Atom substitute(Substitution substitution) {
			return this;
		}
		
		@Override
		public Bindings unify(Formula other, Bindings bindings) {
			if(this.equals(other, bindings))
				return bindings;
			else
				return null;
		}
		
		@Override
		public boolean isTrue(State state) {
			return false;
		}

		@Override
		public void makeTrue(MutableState state) {
			throw new IllegalStateException(this + " can never be made true.");
		}

		@Override
		public Atom negate() {
			return TRUE;
		}
	};
}
