package edu.uky.ai.logic;

import edu.uky.ai.util.ImmutableArray;

/**
 * An implication, or logical IF statement, is defined by the truth value of
 * its two parts, the antecedent and the consequent. It is always true, except
 * when the antecedent is true and the consequent is false.
 * 
 * @author Stephen G. Ware
 */
public class Implication extends BooleanProposition {
	
	/** The word used to express an implication */
	public static final String IMPLICATION_PREDICATE = "if";

	/** The first part of the implication */
	public final Proposition antecedent;
	
	/** The second part of the implication, which is implied by the antecedent  */
	public final Proposition consequent;
	
	/**
	 * Constructs an implication from a given antecedent and consequent.
	 * 
	 * @param antecedent the first part of the implication
	 * @param consequent the second part of the implication which is implied
	 * by the antecedent
	 */
	public Implication(Proposition antecedent, Proposition consequent) {
		super(new ImmutableArray<>(new Proposition[] { antecedent, consequent }));
		this.antecedent = antecedent;
		this.consequent = consequent;
	}
	
	@Override
	public boolean equals(Object other) {
		return other instanceof Implication && argumentsEqual(other);
	}
	
	@Override
	public int hashCode() {
		return Utilities.hashCode(IMPLICATION_PREDICATE, arguments);
	}
	
	@Override
	public String toString() {
		return Utilities.toString(IMPLICATION_PREDICATE, arguments);
	}	

	@Override
	public Proposition substitute(Substitution substitution) {
		ImmutableArray<Proposition> arguments = substituteArguments(substitution);
		if(arguments == this.arguments)
			return (Proposition) substitution.get(this);
		else
			return (Proposition) substitution.get(new Implication(arguments.get(0), arguments.get(1)));
	}
	
	@Override
	public Bindings unify(Formula other, Bindings bindings) {
		if(other instanceof Implication)
			return unifyArguments(other, bindings);
		else
			return null;
	}

	@Override
	public boolean isTrue(State state) {
		return simplify().isTrue(state);
	}

	@Override
	public void makeTrue(MutableState state) {
		simplify().makeTrue(state);
	}

	@Override
	public Proposition simplify() {
		return new Disjunction(antecedent.negate(), consequent);
	}

	@Override
	public Proposition negate() {
		return simplify().negate();
	}

	@Override
	public Proposition toCNF() {
		return simplify().toCNF();
	}

	@Override
	public Proposition toDNF() {
		return simplify().toDNF();
	}
}
